<?php
namespace App\Http\Controllers;

use App\Models\Channel;
use App\Models\ChannelSubscriber;
use App\Models\NewsLanguage;
use App\Models\NewsLanguageSubscriber;
use App\Models\Post;
use App\Models\Setting;
use App\Models\Topic;
use App\Models\TopicFollower;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SearchPostController extends Controller
{
    const TIME_FORMATE = 'Y-m-d H:i';
    public function search(Request $request)
    {
        $userId = Auth::user()->id ?? 0;

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                // If user selected a language, use it (even if not active)
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                // If not selected, use the first active language
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage ? collect([$defaultActiveLanguage->id]) : collect();
            }
        }

        $searchQuery = $request->input('search');
        $channels    = $request->input('channel');
        $topics      = $request->input('topic');
        $filter      = $request->input('filter');

        $channel_ids = ChannelSubscriber::where('user_id', Auth::user()->id ?? 0)->pluck('channel_id')->toArray();
        $topic_ids   = TopicFollower::where('user_id', Auth::user()->id ?? 0)->pluck('topic_id')->toArray();

        $getPosts = Post::select(
            'posts.id',
            'posts.slug',
            'posts.image',
            'posts.comment',
            'channels.name as channel_name',
            'channels.logo as channel_logo',
            'channels.slug as channel_slug',
            'topics.name as topic_name',
            'topics.slug as topic_slug',
            'posts.title',
            'posts.favorite',
            'posts.description',
            'posts.status',
            'posts.publish_date',
            'posts.view_count',
            'posts.type',
            'posts.video_thumb',
            'posts.pubdate',
            'posts.reaction',
        )
            ->where('posts.status', 'active')
            ->join('channels', 'posts.channel_id', '=', 'channels.id')
            ->leftjoin('topics', 'posts.topic_id', '=', 'topics.id');

        if ($subscribedLanguageIds->isNotEmpty()) {
            $getPosts->whereIn('posts.news_language_id', $subscribedLanguageIds);
        }
        if ($searchQuery) {
            $getPosts->where(function ($subQuery) use ($searchQuery) {
                $subQuery->where('posts.slug', 'LIKE', "%$searchQuery%")
                    ->orWhere('posts.title', 'LIKE', "%$searchQuery%")
                    ->orWhere('channels.slug', 'LIKE', "%$searchQuery%")
                    ->orWhere('channels.name', 'LIKE', "%$searchQuery%")
                    ->orWhere('topics.slug', 'LIKE', "%$searchQuery%")
                    ->orWhere('topics.name', 'LIKE', "%$searchQuery%");
            });
        }
        if (! empty($channels)) {
            $getPosts->whereIn('channels.slug', $channels);
        }
        if (! empty($topics)) {
            $getPosts->whereIn('topics.slug', $topics);
        }

        if ($filter == "most-read") {
            $getPosts->where('publish_date', '>', now()->subDays(7)->endOfDay())->orderBy('posts.view_count', 'DESC');
        } elseif ($filter == "most-liked") {

            $getPosts->orderBy('posts.favorite', 'DESC');
        } elseif ($filter == "most-recent") {

            $getPosts->orderBy('posts.publish_date', 'DESC');
        } elseif ($filter == "channels-followed") {

            $getPosts->whereIn('posts.channel_id', $channel_ids);
        } elseif ($filter == "topics-followed") {

            $getPosts->whereIn('posts.topic_id', $topic_ids);
        } else {
            $getPosts->orderBy('posts.publish_date', 'DESC');
        }

        $getPosts     = $getPosts->paginate(15)->withQueryString();
        $channels     = Channel::select('id', 'name', 'slug')->where('status', 'active')->get();
        $topics       = Topic::select('id', 'name', 'slug')->where('status', 'active')->get();
        $post_label   = Setting::get()->where('name', 'news_label_place_holder')->first();
        $defaultImage = Setting::get()->where('name', 'default_image')->first();

        foreach ($getPosts as $post) {

            $post->image = $post->image ?? url($defaultImage->value ?? 'public/front_end/classic/images/default/post-placeholder.jpg');
            if ($post->publish_date) {
                $post->publish_date_news = Carbon::parse($post->pubdate)->format(self::TIME_FORMATE);
                $post->publish_date      = Carbon::parse($post->publish_date)->diffForHumans();
            } else {
                $post->pubdate = Carbon::parse($post->pubdate)->diffForHumans();
            }
        }

        $title = $searchQuery ?? isset($post_label->value) ? $post_label->value : "";
        $theme = getTheme();

        return view('front_end/' . $theme . '/pages/search-result', compact('getPosts', 'title', 'searchQuery', 'post_label', 'channels', 'topics', 'theme'));
    }

    public function autocomplete(Request $request)
    {
        /* ----------------------------------------------------
        STEP 1: Detect user selected news language
    ---------------------------------------------------- */
        $userId = Auth::user()->id ?? 0;

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)
                ->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage
                    ? collect([$defaultActiveLanguage->id])
                    : collect();
            }
        }

        /* ----------------------------------------------------
        STEP 2: Apply Search
    ---------------------------------------------------- */
        $searchQuery = $request->input('search');

        /* ------------------- POSTS -------------------------- */
        $posts = Post::select('title', 'slug', 'image')
            ->when($subscribedLanguageIds->isNotEmpty(), function ($q) use ($subscribedLanguageIds) {
                return $q->whereIn('news_language_id', $subscribedLanguageIds);
            })
            ->when($searchQuery, function ($q) use ($searchQuery) {
                return $q->where('title', 'LIKE', "$searchQuery%");
            })
            ->orderBy('publish_date', 'desc');

        /* ------------------- TOPICS ------------------------- */
        $topics = Topic::selectRaw('name as title, slug, "" as image')
            ->when($subscribedLanguageIds->isNotEmpty(), function ($q) use ($subscribedLanguageIds) {
                return $q->whereIn('news_language_id', $subscribedLanguageIds);
            })
            ->when($searchQuery, function ($q) use ($searchQuery) {
                return $q->where('name', 'LIKE', "$searchQuery%");
            });

        /* ------------------- CHANNELS ------------------------ */
        $channels = Channel::selectRaw('name as title, slug, "" as image')
            ->when($subscribedLanguageIds->isNotEmpty(), function ($q) use ($subscribedLanguageIds) {
                return $q->whereIn('news_language_id', $subscribedLanguageIds);
            })
            ->when($searchQuery, function ($q) use ($searchQuery) {
                return $q->where('name', 'LIKE', "$searchQuery%");
            });

        /* ----------------------------------------------------
        STEP 3: Combine & Format
    ---------------------------------------------------- */
        $combinedQuery = $posts->union($channels)->union($topics);

        $results = $combinedQuery->get();

        $formattedResults = $results->map(function ($item) {
            return [
                'title' => $item->title,
                'url'   => isset($item->slug) ? route('posts.show', $item->slug) : null,
                'image' => $item->image,
            ];
        });

        return response()->json($formattedResults);
    }

    public function getChannel($id)
    {
        $channel = Channel::select('id', 'name', 'logo', 'follow_count')->find($id);

        if ($channel) {
            $channel->channel_logo = url('storage/images/' . $channel->logo);
            return response()->json($channel);
        }

        return response()->json(['error' => 'Channel not found'], 404);
    }
}
